#include <iostream>
#include <stdexcept>

// ---- Template Selection Helper
  
template<bool Condition, class T = void>
struct enable_if {
  typedef T type;
};

template<class T>
struct enable_if<false, T> {};
  
// ---- Meta Assertions

#define STATIC_ASSERT(x) static const int STATIC_ASSERT_ ## __COUNT__ [(x)]

// ---- Runtime Error

struct out_of_range : virtual std::out_of_range {
  out_of_range() : std::out_of_range("value out of range") {}
};

// ---- Our Actual 'restricted' type

template<class T, T Min, T Max>
struct restricted {
  STATIC_ASSERT((Min < Max));
  
  T value;
  
  explicit restricted(T o)
    : value(o) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
  
  template<T OMin, T OMax>
  restricted(restricted<T, OMin, OMax> o
    , typename enable_if< OMin <= Max && Min <= OMax >::type* = 0
  )
    : value(o.value) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }

  //
  // We are used that we can assign values.  Like:
  //  b = c;
  //
  // This does not yet work.  We implement an
  // operator=, with the same SFINAE trick by using
  // enable_if we used earlier to provide compile-time checks
  // where possible.
  //
  // But this time we cannot add a dummy parameter,
  // as operator definitions have a fixed signatures.
  //
  // You may remember that earlier we just specified the
  // expression, but not the type argument of enable_if,
  // which defaulted to void.  Now we will specify the
  // type argument, too, so that we can use enable_if
  // in the function argument list, and still write
  // a functional operator=:
  //
  template<T OMin, T OMax>
  restricted& operator=(
    typename enable_if< OMin <= Max && Min <= OMax,
      restricted<int, OMin, OMax> >::type) {
    if(o.value < Min || Max < o.value) { throw out_of_range(); }
    value = o.value;
  }
};

// ---- How to use it

int main() {
  restricted<int, 0, 10> i(5);
  
  restricted<int, -10, 10> j(1);
  // The assignment now compiles, and provides the
  // same compile-time checks as the constructor:
  j = i;
  
  // TODO: Try:
  //restricted<int, -10, -5> j = i;
}
